

;*****************************************************************************************
;	Mega 128 Scope Controller Program
;	(c) Martin Cibulski
;****************************************************************************************

.EQU	LCD_PAUSE_LONG	= 127
.EQU	LCD_PAUSE_SHORT	= 31

;*****************************************************************************************
.DSEG

mnu_parent:	.byte	2
mnu_child:	.byte	2
mnu_gpar_index:	.byte	2
mnu_par_index:	.byte	2
mnu_chi_index:	.byte	2
mnu_index_step:	.byte	2


lcd_line_1:	.byte	LCD_LENGTH		;buffer for upper display line
lcd_index_1:	.byte	1
lcd_line_2:	.byte	LCD_LENGTH		;buffer for lower display line
lcd_index_2:	.byte	1

lcd_mode:	.byte	1		;LCD_MODE_MOVE or LCD_MODE_MENU
lcd_speed:	.byte	1		;1-4
lcd_lkey_menu:	.byte	1		;previous status of UP/DOWNN/RIGHT/LEFT keys
lcd_lkey_sp:	.byte	1		;                   FAST/SLOW keys
lcd_lkey_mode:	.byte	1		;                   MENU/MOVE keys
lcd_keybits:	.byte	1

lcd_pause:	.byte	1		;timer for speed messages (1 sec)
lcd_rep_del:	.byte	1
lcd_rep_acc:	.byte	1

;*****************************************************************************************
.CSEG

mnu_init:
	ldi	r16	,low(men_root<<1)
	sts	mnu_parent	,r16
	ldi	r16	,high(men_root<<1)
	sts	mnu_parent+1	,r16

	ldi	r16	,low(men_object<<1)
	sts	mnu_child	,r16
	ldi	r16	,high(men_object<<1)
	sts	mnu_child+1	,r16

	clr	r16
	sts	mnu_gpar_index	,r16
	sts	mnu_gpar_index+1	,r16
	sts	mnu_par_index	,r16
	sts	mnu_par_index+1	,r16
	sts	mnu_chi_index	,r16
	sts	mnu_chi_index+1	,r16

	ldi	r16	,1
	sts	mnu_index_step	,r16
	clr	r16
	sts	mnu_index_step+1	,r16

;*****************************************************************************************
mnu_show:
	clr	r16
	sts	lcd_pause	,r16

	lds	r16	,lcd_mode		;show menu or position ?
	cpi	r16	,LCD_MODE_MOVE
	breq	mnu_show_position

	lds	zl	,mnu_parent		;parent menu entry
	lds	zh	,mnu_parent+1
	adiw	zh:zl	,MNU_ADRESS
	lpm	r16	,Z+
	lpm	r17	,Z+
	tst	r16			;function adress 
	brne	mnu_show_dynamic1

	tst	r17			;no function adress,
	breq	mnu_show_static1			;show static text only

mnu_show_dynamic1:
	mov	zl	,r16
	mov	zh	,r17
	adiw	zh:zl	,MNU_CALL_SHOW1		;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brts	mnu_show_2			;function didn't print, show static text

mnu_show_static1:
	lds	zl	,mnu_parent
	lds	zh	,mnu_parent+1
	rcall	lcd_print_z1

mnu_show_2:
	lds	zl	,mnu_child		;child menu entry
	lds	zh	,mnu_child+1
	adiw	zh:zl	,MNU_ADRESS
	lpm	r16	,Z+
	lpm	r17	,Z+
	tst	r16
	brne	mnu_show_dynamic2

	tst	r17
	breq	mnu_show_static2

mnu_show_dynamic2:
	mov	zl	,r16
	mov	zh	,r17
	adiw	zh:zl	,MNU_CALL_SHOW2		;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brtc	mnu_show_static2

	ret

mnu_show_static2:
	lds	zl	,mnu_child
	lds	zh	,mnu_child+1
	rjmp	lcd_print_z2

mnu_show_position:
	ldi	xl	,low(cor_recttext)
	ldi	xh	,high(cor_recttext)
	rcall	lcd_print_x1

	ldi	xl	,low(cor_decltext)
	ldi	xh	,high(cor_decltext)
	rjmp	lcd_print_x2


;*****************************************************************************************
;

mnu_out:
	lds	zl	,mnu_parent		;pointer to parent node
	lds	zh	,mnu_parent+1
	adiw	zh:zl	,MNU_PARENT_PTR
	lpm	r16	,Z+		;check pointer to parent's parent
	lpm	r17	,Z+
	or	r16	,r17
	brne	mnu_out_01			;not null, ok
	ret
mnu_out_01:
	lds	zl	,mnu_parent		;pointer to parent node
	lds	zh	,mnu_parent+1
	sts	mnu_child	,zl		;set as new child
	sts	mnu_child+1	,zh
	adiw	zh:zl	,MNU_PARENT_PTR
	lpm	r16	,Z+		;parent's parent node
	lpm	r17	,Z+
	lsl	r16			;convert to byte pointer
	rol	r17
	sts	mnu_parent	,r16		;set as new parent
	sts	mnu_parent+1	,r17
	lds	r16	,mnu_gpar_index
	lds	r17	,mnu_gpar_index+1
	sts	mnu_par_index	,r16
	sts	mnu_par_index+1	,r17
	lds	r16	,mnu_par_index
	lds	r17	,mnu_par_index+1
	sts	mnu_chi_index	,r16
	sts	mnu_chi_index+1	,r17
	clr	r16
	sts	mnu_gpar_index	,r16
	sts	mnu_gpar_index+1	,r16
	rjmp	mnu_show

;*****************************************************************************************
;step into a menu subtree

mnu_in:
	lds	zl	,mnu_child		;get function adress of subtree
	lds	zh	,mnu_child+1		;if function = 0
	adiw	zh:zl	,MNU_ADRESS		;it is a subtree having child entries
	lpm	r16	,Z+
	lpm	r17	,Z+
	mov	r0	,r16
	or	r0	,r17
	breq	mnu_in_into			;function adress = 0, step into subtree

	mov	zl	,r16		;otherwise call function
	mov	zh	,r17
	adiw	zh:zl	,MNU_CALL_EXEC		;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brts	mnu_in_end			;all done, if not step into subtree anyway

mnu_in_into:
	lds	r16	,mnu_child
	lds	r17	,mnu_child+1

	ldi	yl	,low(men_root<<1)
	ldi	yh	,high(men_root<<1)

mnu_in_loop:
	mov	zl	,yl
	mov	zh	,yh
	adiw	zh:zl	,MNU_PARENT_PTR
	lpm	r18	,Z+
	lpm	r19	,Z+
	cpi	r18	,0xFF
	brne	mnu_in_no_tab_end

	cpi	r19	,0xFF
	brne	mnu_in_no_tab_end

mnu_in_end:
	ret

mnu_in_no_tab_end:
	lsl	r18
	rol	r19

	cp	r16	,r18
	brne	mnu_in_next

	cp	r17	,r19
	breq	mnu_in_found

mnu_in_next:
	adiw	yh:yl	,MNU_ENTRY_SIZE
	rjmp	mnu_in_loop

mnu_in_found:
	sts	mnu_parent	,r16
	sts	mnu_parent+1	,r17
	sts	mnu_child	,yl
	sts	mnu_child+1	,yh
	lds	r16	,mnu_par_index
	lds	r17	,mnu_par_index+1
	sts	mnu_gpar_index	,r16
	sts	mnu_gpar_index+1	,r17
	lds	r16	,mnu_chi_index
	lds	r17	,mnu_chi_index+1
	sts	mnu_par_index	,r16
	sts	mnu_par_index+1	,r17

	movw	zh:zl	,yh:yl		;get child items start index
	adiw	zh:zl	,MNU_ADRESS
	lpm	r16	,Z+
	lpm	r17	,Z+
	mov	r0	,r16
	or	r0	,r17
	breq	mnu_in_child_index		;no func, take 0 as index

	movw	zh:zl	,r17:r16
	adiw	zh:zl	,MNU_CALL_STARTIND	;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brts	mnu_in_child_index		;index set, if not index=0

	clr	r16
	clr	r17

mnu_in_child_index:
	sts	mnu_chi_index	,r16
	sts	mnu_chi_index+1	,r17
	rjmp	mnu_show

;*****************************************************************************************
mnu_prev:
	lds	zl	,mnu_child		;if (index < child->max_no)
	lds	zh	,mnu_child+1

	adiw	zh:zl	,MNU_ADRESS
	lpm	r16	,Z+
	lpm	r17	,Z+
	mov	r18	,r16
	or	r18	,r17
	breq	mnu_prev_01

	mov	zl	,r16		;otherwise call function
	mov	zh	,r17
	adiw	zh:zl	,MNU_CALL_MAX_IND		;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brts	mnu_prev_01			;all done, if not set max index to zero

	clr	r16			;max index 0x0000
	clr	r17
	clr	r18			;no wrap around

mnu_prev_01:
	lds	r24	,mnu_chi_index		;index > 0 ?
	lds	r25	,mnu_chi_index+1
	mov	r0	,r24
	or	r0	,r25
	brne	mnu_prev_step_backwards		;yes, step back

	tst	r18			;no, wrap around allowed ?
	brpl	mnu_prev_ptr			;no, go to menu item before

	movw	r25:r24	,r17:r16		;yes, wrap around to max_index
	rjmp	mnu_prev_index_inrange

mnu_prev_step_backwards:
	lds	r0	,mnu_index_step		;index = index + step
	lds	r1	,mnu_index_step+1
	sub	r24	,r0
	sbc	r25	,r1
	brcc	mnu_prev_index_inrange		;no underflow, save new index

	clr	r24			;underflow, set index = 0
	clr	r25

mnu_prev_index_inrange:
	sts	mnu_chi_index	,r24
	sts	mnu_chi_index+1	,r25
	rjmp	mnu_show

mnu_prev_ptr:
	lds	zl	,mnu_child		;
	lds	zh	,mnu_child+1
	sbiw	zh:zl	,MNU_ENTRY_SIZE - MNU_PARENT_PTR
	lpm	r16	,Z+
	lpm	r17	,Z+
	lsl	r16
	rol	r17
	lds	r18	,mnu_parent
	lds	r19	,mnu_parent+1
	cp	r16	,r18
	cpc	r17	,r19
	brne	mnu_prev_index_inrange

	lds	zl	,mnu_child		;   child --
	lds	zh	,mnu_child+1
	sbiw	zh:zl	,MNU_ENTRY_SIZE
	sts	mnu_child	,zl
	sts	mnu_child+1	,zh

	clr	r16			;   index = 0
	sts	mnu_chi_index	,r16
	sts	mnu_chi_index+1	,r16
	rjmp	mnu_show


;*****************************************************************************************

mnu_next:
	lds	zl	,mnu_child		;if (index < child->max_no)
	lds	zh	,mnu_child+1

	adiw	zh:zl	,MNU_ADRESS
	lpm	r16	,Z+
	lpm	r17	,Z+
	mov	r18	,r16
	or	r18	,r17
	breq	mnu_next_01

	mov	zl	,r16		;otherwise call function
	mov	zh	,r17
	adiw	zh:zl	,MNU_CALL_MAX_IND		;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brts	mnu_next_01			;all done, if not set max index to zero

	clr	r16			;max index 0x0000
	clr	r17
	clr	r18			;no wrap around

mnu_next_01:
	lds	r24	,mnu_chi_index		;index < max_index ?
	lds	r25	,mnu_chi_index+1
	cp	r24	,r16
	cpc	r25	,r17
	brcs	mnu_next_step_forward		;yes, step forward

	tst	r18			;no, wrap around allowed ?
	brpl	mnu_next_ptr			;no, go to next menu item

	clr	r24			;yes, wrap around to 0
	clr	r25
	rjmp	mnu_next_index_inrange

mnu_next_step_forward:
	lds	r0	,mnu_index_step		;index = index + step
	lds	r1	,mnu_index_step+1
	add	r24	,r0
	adc	r25	,r1
	cp	r16	,r24		;out of range ?
	cpc	r17	,r25
	brcc	mnu_next_index_inrange		;no

	mov	r24	,r16		;set max index
	mov	r25	,r17

mnu_next_index_inrange:
	sts	mnu_chi_index	,r24
	sts	mnu_chi_index+1	,r25
	rjmp	mnu_show

mnu_next_ptr:
	lds	zl	,mnu_child		;
	lds	zh	,mnu_child+1
	adiw	zh:zl	,MNU_ENTRY_SIZE + MNU_PARENT_PTR
	lpm	r16	,Z+
	lpm	r17	,Z+
	lsl	r16
	rol	r17
	lds	r18	,mnu_parent
	lds	r19	,mnu_parent+1
	cp	r16	,r18
	cpc	r17	,r19
	brne	mnu_next_index_inrange

	lds	zl	,mnu_child		;   child ++
	lds	zh	,mnu_child+1
	adiw	zh:zl	,MNU_ENTRY_SIZE
	sts	mnu_child	,zl
	sts	mnu_child+1	,zh

	clr	r16			;   index = 0
	sts	mnu_chi_index	,r16
	sts	mnu_chi_index+1	,r16		;}
	rjmp	mnu_show


;*****************************************************************************************


.equ	LCD_BUSY	= 7

.equ	LCD_CLEAR	= 0b00000001

.equ	LCD_HOME	= 0b00000010
.equ	LCD_HOME2	= 0b10000000 + 0x40

.equ	LCD_ENTRY	= 0b00000100
.equ	LCD_ENTRY_INC	= 0b00000010
.equ	LCD_ENTRY_DEC	= 0b00000000
.equ	LCD_ENTRY_SHIFT	= 0b00000001
.equ	LCD_ENTRY_NOSHIFT	= 0b00000000

.equ	LCD_DISP	= 0b00001000
.equ	LCD_DISP_OFF	= 0b00000000
.equ	LCD_DISP_ON	= 0b00000100
.equ	LCD_DISP_NOCURSOR	= 0b00000000
.equ	LCD_DISP_CURSOR	= 0b00000010
.equ	LCD_DISP_NOBLINK	= 0b00000000
.equ	LCD_DISP_BLINK	= 0b00000001

.equ	LCD_SET	= 0b00100000
.equ	LCD_SET_4BIT	= 0b00000000
.equ	LCD_SET_8BIT	= 0b00010000
.equ	LCD_SET_1LINE	= 0b00000000
.equ	LCD_SET_2LINE	= 0b00001000
.equ	LCD_SET_FONT7	= 0b00000000
.equ	LCD_SET_FONT10	= 0b00000100

;*****************************************************************************************

.equ	LCD_MODE_MENU	= 1
.equ	LCD_MODE_MOVE	= 0

.equ	LCD_KEY_LEFT	= 0b11011111
.equ	LCD_KEY_RIGHT	= 0b10111111
.equ	LCD_KEY_UP	= 0b01111111
.equ	LCD_KEY_DOWN	= 0b11101111

.equ	LCD_KEY_FAST	= 0b11110111
.equ	LCD_KEY_SLOW	= 0b11111011

.equ	LCD_KEY_MENU	= 0b11111101
.equ	LCD_KEY_MOVE	= 0b11111110
.equ	LCD_KEYENTRY_SIZE	= 4

.equ	LCD_REPEAT_SHORT	= 18
.equ	LCD_REPEAT_LONG	= 100

;*****************************************************************************************

.CSEG

lcd_init:
	sbi	LCD_CTRLPORT	,LCD_PIN_KEYS		;disable handbox keys
	cbi	LCD_CTRLPORT	,LCD_PIN_RW		;display as input
	cbi	LCD_CTRLPORT	,LCD_PIN_E		;enable = 0
	ldi	r16	,0xFF		;controlport as output
	out	LCD_CTRLDDR	,r16

	ldi	r16	,0x00		;dataport as input
	out	LCD_DATADDR	,r16
	ldi	r16	,0xFF		;pullup resistors on
	out	LCD_DATAPORT	,r16

	rcall	lcd_wait_long

	ldi	r16	,0b00110000
	cbi	LCD_CTRLPORT	,LCD_PIN_RS		;register 0
	rcall	lcd_write
	rcall	lcd_wait_long			;minimum 15 ms

	ldi	r16	,0b00110000
	cbi	LCD_CTRLPORT	,LCD_PIN_RS		;register 0
	rcall	lcd_write
	rcall	lcd_wait_long			;minimum 4.1 ms

	ldi	r16	,0b00110000
	cbi	LCD_CTRLPORT	,LCD_PIN_RS		;register 0
	rcall	lcd_write
	rcall	lcd_wait_long			;minimum 0.1 ms

	ldi	r16	,LCD_SET + LCD_SET_8BIT + LCD_SET_2LINE + LCD_SET_FONT7
	rcall	lcd_write_command

	ldi	r16	,LCD_DISP + LCD_DISP_ON + LCD_DISP_NOCURSOR + LCD_DISP_NOBLINK
	rcall	lcd_write_command

	ldi	r16	,LCD_CLEAR
	rcall	lcd_write_command

	ldi	r16	,LCD_ENTRY + LCD_ENTRY_INC + LCD_ENTRY_NOSHIFT
	rcall	lcd_write_command

	ldi	r16	,LCD_MODE_MOVE
	sts	lcd_mode	,r16

	ldi	r16	,1
	sts	lcd_speed	,r16

	ldi	r16	,0b11111111
	sts	lcd_lkey_mode	,r16
	sts	lcd_lkey_menu	,r16
	sts	lcd_lkey_sp	,r16

	ret

;*****************************************************************************************

lcd_print_z1:
	ldi	xl	,low(lcd_line_1)
	ldi	xh	,high(lcd_line_1)
	ldi	r16	,7
lcd_print_z1_loop:
	lpm	r0	,Z+
	st	X+	,r0
	dec	r16
	brpl	lcd_print_z1_loop

	sts	lcd_index_1	,r16
	ret

lcd_print_z2:
	ldi	xl	,low(lcd_line_2)
	ldi	xh	,high(lcd_line_2)
	ldi	r16	,7
lcd_print_z2_loop:
	lpm	r0	,Z+
	st	X+	,r0
	dec	r16
	brpl	lcd_print_z2_loop

	sts	lcd_index_2	,r16
	ret

;*****************************************************************************************

lcd_print_x1:
	ldi	zl	,low(lcd_line_1)
	ldi	zh	,high(lcd_line_1)
	ldi	r16	,7
lcd_print_x1_loop:
	ld	r0	,X+
	st	Z+	,r0
	dec	r16
	brpl	lcd_print_x1_loop

	sts	lcd_index_1	,r16
	ret

lcd_print_x2:
	ldi	zl	,low(lcd_line_2)
	ldi	zh	,high(lcd_line_2)
	ldi	r16	,7
lcd_print_x2_loop:
	ld	r0	,X+
	st	Z+	,r0
	dec	r16
	brpl	lcd_print_x2_loop

	sts	lcd_index_2	,r16
	ret

;*****************************************************************************************

lcd_irq:
	in	r16	,LCD_DATAPINS		;read signals from keyboard
	sts	lcd_keybits	,r16		;store
	sbi	LCD_CTRLPORT	,LCD_PIN_KEYS		;disable keys for using the LCD

	lds	r16	,lcd_pause		;temporary message active ?
	tst	r16
	breq	lcd_irq_no_show			;no

	dec	r16			;timer--
	sts	lcd_pause	,r16		;save timer
	brne	lcd_irq_no_show			;still not zero

	rcall	mnu_show			;if zero, show menu/position again
	clr	r16
	sts	cor_new_position	,r16

lcd_irq_no_show:
	rcall	lcd_read_busy			;LCD busy ?
	brtc	lcd_irq_lcd_available		;no, write next character

	rjmp	lcd_read_keys			;yes

lcd_irq_lcd_available:
	lds	r16	,lcd_index_1		;cursor 1
	cpi	r16	,LCD_LENGTH		;output completed ?
	breq	lcd_irq_line_2			;yes, check line 2

	cpi	r16	,0xFF
	brne	lcd_irq_write_1

	ldi	r16	,LCD_HOME
	rcall	lcd_write_command
	ldi	r16	,0
	sts	lcd_index_1	,r16
	rjmp	lcd_read_keys

lcd_irq_write_1:
	inc	r16			;print character in LCD line 1
	sts	lcd_index_1	,r16
	ldi	xl	,low(lcd_line_1 - 1)
	ldi	xh	,high(lcd_line_1 - 1)
	add	xl	,r16
	clr	r16
	adc	xh	,r16
	ld	r16	,X
	rcall	lcd_write_char
	rjmp	lcd_read_keys

lcd_irq_line_2:
	lds	r16	,lcd_index_2		;cursor 2
	cpi	r16	,LCD_LENGTH		;output completed ?
	breq	lcd_irq_check_new_pos		;yes, display new position

	cpi	r16	,0xFF		;just initial
	brne	lcd_irq_write_2

	ldi	r16	,LCD_HOME2
	rcall	lcd_write_command
	ldi	r16	,0
	sts	lcd_index_2	,r16
	rjmp	lcd_read_keys

lcd_irq_write_2:
	inc	r16			;print character in LCD line 2
	sts	lcd_index_2	,r16
	ldi	xl	,low(lcd_line_2 - 1)
	ldi	xh	,high(lcd_line_2 - 1)
	add	xl	,r16
	clr	r16
	adc	xh	,r16
	ld	r16	,X
	rcall	lcd_write_char

lcd_irq_check_new_pos:
	lds	r16	,lcd_pause		;temporary message active ?
	tst	r16
	brne	lcd_read_keys			;yes

	lds	r16	,lcd_mode		;MOVE mode ?
	cpi	r16	,LCD_MODE_MOVE
	brne	lcd_read_keys			;no

	lds	r16	,cor_new_position		;new position to display ?
	tst	r16
	breq	lcd_read_keys			;no

	rcall	mnu_show			;show new position
	clr	r16
	sts	cor_new_position	,r16

	ldi	r16	,LCD_PAUSE_SHORT
	sts	lcd_pause	,r16

lcd_read_keys:

	lds	r17	,lcd_mode
	cpi	r17	,LCD_MODE_MENU
	brne	lcd_read_keys_nomenu

	ldi	yl	,low(mot_a)		;Motor A
	ldi	yh	,high(mot_a)
	call	mot_speed_null

	ldi	yl	,low(mot_b)		;Motor B
	ldi	yh	,high(mot_b)
	call	mot_speed_null

	lds	r16	,lcd_keybits
	ori	r16	,LCD_KEY_UP & LCD_KEY_DOWN & LCD_KEY_RIGHT & LCD_KEY_LEFT
	lds	r19	,lcd_lkey_menu
	set
	rcall	lcd_key_handler
	sts	lcd_lkey_menu	,r19

lcd_read_keys_nomenu:
	lds	r17	,lcd_mode
	cpi	r17	,LCD_MODE_MOVE
	brne	lcd_read_keys_no_move

	ldi	yl	,low(mot_b)		;Motor B
	ldi	yh	,high(mot_b)

	lds	r16	,lcd_keybits
	ori	r16	,LCD_KEY_UP & LCD_KEY_DOWN
	cpi	r16	,LCD_KEY_UP
	brne	lcd_read_keys_no_up

	rcall	lcd_move_plus
	rjmp	lcd_read_keys_leftright

lcd_read_keys_no_up:
	cpi	r16	,LCD_KEY_DOWN
	brne	lcd_read_keys_no_down

	rcall	lcd_move_minus
	rjmp	lcd_read_keys_leftright

lcd_read_keys_no_down:
	call	mot_speed_null

lcd_read_keys_leftright:
	ldi	yl	,low(mot_a)		;Motor A
	ldi	yh	,high(mot_a)

	lds	r16	,lcd_keybits
	ori	r16	,LCD_KEY_LEFT & LCD_KEY_RIGHT
	cpi	r16	,LCD_KEY_LEFT
	brne	lcd_read_keys_no_left

	rcall	lcd_move_plus
	rjmp	lcd_read_keys_no_move

lcd_read_keys_no_left:
	cpi	r16	,LCD_KEY_RIGHT
	brne	lcd_read_keys_no_right

	rcall	lcd_move_minus
	rjmp	lcd_read_keys_no_move

lcd_read_keys_no_right:
	call	mot_speed_null

lcd_read_keys_no_move:
	lds	r16	,lcd_keybits
	ori	r16	,LCD_KEY_MENU & LCD_KEY_MOVE
	lds	r19	,lcd_lkey_mode
	clt
	rcall	lcd_key_handler
	sts	lcd_lkey_mode	,r19

	lds	r16	,lcd_keybits
	ori	r16	,LCD_KEY_FAST & LCD_KEY_SLOW
	lds	r19	,lcd_lkey_sp
	clt
	rcall	lcd_key_handler
	sts	lcd_lkey_sp	,r19

	ldi	r16	,0x00
	out	LCD_DATADDR	,r16
	ldi	r16	,0xFF
	out	LCD_DATAPORT	,r16
	cbi	LCD_CTRLPORT	,LCD_PIN_RW
	cbi	LCD_CTRLPORT	,LCD_PIN_KEYS
	ret

;*****************************************************************************************
;r16	command

lcd_write_command:
	rcall	lcd_read_busy
	brts	lcd_write_command

	cbi	LCD_CTRLPORT	,LCD_PIN_RS	;register 0
	rjmp	lcd_write


;*****************************************************************************************
;r16	data

lcd_write_char:
	rcall	lcd_read_busy
	brts	lcd_write_char

	sbi	LCD_CTRLPORT	,LCD_PIN_RS	;register 1
	rjmp	lcd_write


;*****************************************************************************************

lcd_wait_long:
.ifdef SIMULATE
	ret
.endif
	push	r0
	push	r1
	clr	r0
	clr	r1
lcd_wait_long_01:
	dec	r0
	nop
	nop
	nop
	brne	lcd_wait_long_01
	dec	r1
	brne	lcd_wait_long_01
	pop	r1
	pop	r0
	ret



;*****************************************************************************************

lcd_read_busy:
	push	r16
	cbi	LCD_CTRLPORT	,LCD_PIN_RS
	rcall	lcd_read

	bst	r16	,LCD_BUSY
	pop	r16
	ret	

;*****************************************************************************************
;r16	byte

lcd_write:
	push	r16

	ldi	r16	,0xFF
	out	LCD_DATADDR	,r16
	cbi	LCD_CTRLPORT	,LCD_PIN_RW

	pop	r16
	out	LCD_DATAPORT	,r16
	sbi	LCD_CTRLPORT	,LCD_PIN_E
	ldi	r16	,100
lcd_write_01:
	dec	r16
	brne	lcd_write_01

	cbi	LCD_CTRLPORT	,LCD_PIN_E
	ldi	r16	,0x00
	out	LCD_DATADDR	,r16
	ret


;*****************************************************************************************

lcd_read:
	ldi	r16	,0x00
	out	LCD_DATADDR	,r16

	sbi	LCD_CTRLPORT	,LCD_PIN_RW
	sbi	LCD_CTRLPORT	,LCD_PIN_E
	ldi	r16	,100
lcd_read_01:
	dec	r16
	brne	lcd_read_01

	in	r16	,LCD_DATAPINS
	cbi	LCD_CTRLPORT	,LCD_PIN_E
	cbi	LCD_CTRLPORT	,LCD_PIN_RW
	ret


;*****************************************************************************************

lcd_move_plus:
	ldi	r16	,TGO_ST_NONE		;disable old tracking target
	sts	tgo_status	,r16

	lds	r16	,lcd_speed
	cpi	r16	,5
	brne	lcd_move_plus_no5

	jmp	mot_speed_plus5

lcd_move_plus_no5:
	cpi	r16	,4
	brne	lcd_move_plus_no4

	jmp	mot_speed_plus4

lcd_move_plus_no4:
	cpi	r16	,3
	brne	lcd_move_plus_no3

	jmp	mot_speed_plus3

lcd_move_plus_no3:
	cpi	r16	,2
	brne	lcd_move_plus_no2

	jmp	mot_speed_plus2

lcd_move_plus_no2:
	jmp	mot_speed_plus1


;*****************************************************************************************

lcd_move_minus:
	ldi	r16	,TGO_ST_NONE		;disable old tracking target
	sts	tgo_status	,r16

	lds	r16	,lcd_speed
	cpi	r16	,5
	brne	lcd_move_minus_no5

	jmp	mot_speed_minus5

lcd_move_minus_no5:
	cpi	r16	,4
	brne	lcd_move_minus_no4

	jmp	mot_speed_minus4

lcd_move_minus_no4:
	cpi	r16	,3
	brne	lcd_move_minus_no3

	jmp	mot_speed_minus3

lcd_move_minus_no3:
	cpi	r16	,2
	brne	lcd_move_minus_no2

	jmp	mot_speed_minus2

lcd_move_minus_no2:
	jmp	mot_speed_minus1


;*****************************************************************************************
;r16	pressed key
;r19	last key 
;T flag	1=autorepeat allowed

lcd_key_handler:

	cp	r16	,r19		;same key pressed
	breq	lcd_key_handler_00		;yes, check if autorepeat allowed

	ldi	r19	,LCD_REPEAT_LONG		;new key pressed
	sts	lcd_rep_del	,r19		;set autorepeat delay
	lsl	r19
	sts	lcd_rep_acc	,r19		;set repeat acceleration delay
	clr	r19
	sts	mnu_index_step+1	,r19		;set autorepeat step = 1
	ldi	r19	,1
	sts	mnu_index_step	,r19
	rjmp	lcd_key_handler_02		;execute menu function

lcd_key_handler_00:
	brtc	lcd_key_handler_norepeat		;no autorepeat allowed


;***********************************************************************
;autorepeat with acceleration

	lds	r16	,lcd_rep_acc		;decrement acceleration delay
	dec	r16
	sts	lcd_rep_acc	,r16
	brne	lcd_khd_no_accel			;not zero, don't accelerate

	lds	zl	,mnu_child		;get maximum stepsize from menu function
	lds	zh	,mnu_child+1
	adiw	zh:zl	,MNU_ADRESS
	lpm	r0	,Z+
	lpm	r1	,Z+
	mov	r16	,r0
	or	r16	,r1
	breq	lcd_khd_no_accel			;no function adress, no acceleration

	mov	zl	,r0		;otherwise call function
	mov	zh	,r1
	adiw	zh:zl	,MNU_CALL_FASTSTEP	;offset in menu item's jump table
	clt
	icall				;call into menu item's jump table
	brtc	lcd_khd_no_accel			;no info, don't accelerate

	lds	r0	,mnu_index_step		;stepsize = stepsize * 2
	lds	r1	,mnu_index_step+1
	lsl	r0
	rol	r1
	sts	mnu_index_step	,r0
	sts	mnu_index_step+1	,r1
	cp	r16	,r0		;stepsize too big ?
	cpc	r17	,r1
	brcc	lcd_khd_01			;no

	sts	mnu_index_step	,r16		;stepsize = max_stepsize
	sts	mnu_index_step+1	,r17

lcd_khd_01:
	ldi	r16	,LCD_REPEAT_LONG		;start delay for next acceleration
	sts	lcd_rep_acc	,r16

lcd_khd_no_accel:
	lds	r16	,lcd_rep_del
	dec	r16
	sts	lcd_rep_del	,r16
	brpl	lcd_key_handler_norepeat

	ldi	r16	,LCD_REPEAT_SHORT
	sts	lcd_rep_del	,r16
	mov	r16	,r19
	rjmp	lcd_key_handler_02

lcd_key_handler_norepeat:
	mov	r16	,r19
	ret


;***********************************************************************

lcd_key_handler_02:
	ldi	yl	,low(lcd_keys_table<<1)
	ldi	yh	,high(lcd_keys_table<<1)
	lds	r17	,lcd_mode
	ldi	r18	,12

lcd_key_handler_loop:
	mov	zl	,yl
	mov	zh	,yh
	lpm	r19	,Z+
	cp	r19	,r17
	brne	lcd_key_handler_next

	lpm	r20	,Z+
	cp	r20	,r16
	brne	lcd_key_handler_next

	lpm	yl	,Z+
	lpm	yh	,Z+
	mov	r18	,yl
	or	r18	,yh
	breq	lcd_key_handler_end

	mov	zl	,yl
	mov	zh	,yh
	push	r16
	icall
	pop	r19
	ret

lcd_key_handler_next:
	adiw	yh:yl	,LCD_KEYENTRY_SIZE
	dec	r18
	brne	lcd_key_handler_loop

lcd_key_handler_end:
	ldi	r19	,0xFF
	ret

lcd_keys_table:
	.db	LCD_MODE_MOVE, LCD_KEY_MENU
	.dw	lcd_modemenu
	.db	LCD_MODE_MOVE, LCD_KEY_MOVE
	.dw	lcd_modemove
	.db	LCD_MODE_MOVE, LCD_KEY_FAST
	.dw	lcd_speed_fast
	.db	LCD_MODE_MOVE, LCD_KEY_SLOW
	.dw	lcd_speed_slow

	.db	LCD_MODE_MENU, LCD_KEY_LEFT
	.dw	mnu_prev
	.db	LCD_MODE_MENU, LCD_KEY_RIGHT
	.dw	mnu_next
	.db	LCD_MODE_MENU, LCD_KEY_UP
	.dw	mnu_out
	.db	LCD_MODE_MENU, LCD_KEY_DOWN
	.dw	mnu_in
	.db	LCD_MODE_MENU, LCD_KEY_MENU
	.dw	lcd_modemenu
	.db	LCD_MODE_MENU, LCD_KEY_MOVE
	.dw	lcd_modemove
	.db	LCD_MODE_MENU, LCD_KEY_FAST
	.dw	lcd_speed_fast
	.db	LCD_MODE_MENU, LCD_KEY_SLOW
	.dw	lcd_speed_slow

;*****************************************************************************************
lcd_modemenu:
	lds	r16	,lcd_mode
	cpi	r16	,LCD_MODE_MENU
	brne	lcd_modemenu_01

	ret

lcd_modemenu_01:
	rcall	mnu_out
	ldi	r16	,LCD_MODE_MENU
	sts	lcd_mode	,r16

	ldi	yl	,low(mot_a)		;Motor A
	ldi	yh	,high(mot_a)
	call	mot_speed_null

	ldi	yl	,low(mot_b)		;Motor B
	ldi	yh	,high(mot_b)
	call	mot_speed_null

	rjmp	mnu_show

;*****************************************************************************************
lcd_modemove:
	ldi	r16	,LCD_MODE_MOVE
	sts	lcd_mode	,r16
	rjmp	mnu_show

;*****************************************************************************************
lcd_speed_fast:
	lds	r16	,lcd_speed
	cpi	r16	,5
	breq	lcd_speed_print

	inc	r16
	sts	lcd_speed	,r16
	rjmp	lcd_speed_print

lcd_speed_slow:
	lds	r16	,lcd_speed
	cpi	r16	,1
	breq	lcd_speed_print

	dec	r16
	sts	lcd_speed	,r16

lcd_speed_print:
	ldi	r16	,LCD_PAUSE_LONG
	sts	lcd_pause	,r16

	ldi	zl	,low(lcd_speed0_msg<<1)
	ldi	zh	,high(lcd_speed0_msg<<1)
	call	lcd_print_z1

	lds	r16	,lcd_speed
	cpi	r16	,1
	brne	lcd_speed_no1

	ldi	zl	,low(lcd_speed1_msg<<1)
	ldi	zh	,high(lcd_speed1_msg<<1)
	jmp	lcd_print_z2

lcd_speed_no1:	cpi	r16	,2
	brne	lcd_speed_no2

	ldi	zl	,low(lcd_speed2_msg<<1)
	ldi	zh	,high(lcd_speed2_msg<<1)
	jmp	lcd_print_z2

lcd_speed_no2:	cpi	r16	,3
	brne	lcd_speed_no3

	ldi	zl	,low(lcd_speed3_msg<<1)
	ldi	zh	,high(lcd_speed3_msg<<1)
	jmp	lcd_print_z2

lcd_speed_no3:	cpi	r16	,4
	brne	lcd_speed_no4

	ldi	zl	,low(lcd_speed4_msg<<1)
	ldi	zh	,high(lcd_speed4_msg<<1)
	jmp	lcd_print_z2

lcd_speed_no4:	cpi	r16	,5
	brne	lcd_speed_no5

	ldi	zl	,low(lcd_speed5_msg<<1)
	ldi	zh	,high(lcd_speed5_msg<<1)
	jmp	lcd_print_z2

lcd_speed_no5:
	ldi	zl	,low(lcd_speedx_msg<<1)
	ldi	zh	,high(lcd_speedx_msg<<1)
	jmp	lcd_print_z2

lcd_speed0_msg:
	.db	"********"
lcd_speed1_msg:
	.db	"SPEED 1 "
lcd_speed2_msg:
	.db	"SPEED 2 "
lcd_speed3_msg:
	.db	"SPEED 3 "
lcd_speed4_msg:
	.db	"SPEED 4 "
lcd_speed5_msg:
	.db	"SPEED 5 "
lcd_speedx_msg:
	.db	"SPEED ? "

;*****************************************************************************************
