

;*****************************************************************************************
;Telescope Mount Controller III
;(c) Martin Cibulski
;
;CPU........: Atmel AVR mega128, 16 MHz
;Stepper ICs: Infineon TCA 3727
;
;(Formatted for editor tab witdth of 18 characters)
;
;****************************************************************************************

;TO DO LIST
;----------
;Message when Z3 cannot be found
;label tab for object_db.asm


;****************************************************************************************

.LISTMAC
.INCLUDE	"m128def.inc"

;.equ SIMULATE = 1

;*****************************************************************************************

.DSEG
.ORG	0x100
ram_start:

;*****************************************************************************************
;Vector Table (Reset, Interrupts)
;
;Timer 1	8889 per second	Motor A microstepping / halfstepping
;Timer 3	8889 per second	Motor B microstepping / halfstepping
;Timer 0	128 per second	Real time clock (external 32768 Hz quartz)
;		Speed ramping of both motors
;		Handbox menu (key input and LCD output)
;USART0	receive complete	Serial communication

.CSEG
.ORG	0x0000

	jmp	main			;Reset

	jmp	to_reti			;External Tnterrupt 0
	jmp	to_reti			;External Tnterrupt 1
	jmp	to_reti			;External Tnterrupt 2
	jmp	to_reti			;External Tnterrupt 3
	jmp	to_reti			;External Tnterrupt 4
	jmp	to_reti			;External Tnterrupt 5
	jmp	to_reti			;External Tnterrupt 6
	jmp	to_reti			;External Tnterrupt 7

	jmp	to_reti			;Timer/Counter 2 Compare Match
	jmp	to_reti			;Timer/Counter 2 Overflow

	jmp	to_reti			;Timer/Counter 1 Capture Event
	jmp	irq_timer_1			;Timer/Counter 1 Compare Match A
	jmp	to_reti			;Timer/Counter 1 Compare Match B
	jmp	to_reti			;Timer/Counter 1 Overflow

	jmp	irq_timer_0			;Timer/Counter 0 Compare Match
	jmp	to_reti			;Timer/Counter 0 Overflow

	jmp	to_reti			;SPI Serial Transfer Complete

	jmp	usart0_rx			;USART0, RX Complete
	jmp	to_reti			;USART0, Data Register Empty
	jmp	to_reti			;USART0, RX Complete

	jmp	to_reti			;ADC conversion complete
	jmp	to_reti			;EEPROM ready
	jmp	to_reti			;Analog comparator

	jmp	to_reti			;Timer/Counter 1 Compare Match C
	jmp	to_reti			;Timer/Counter 3 Capture Event
	jmp	irq_timer_3			;Timer/Counter 3 Compare Match A
	jmp	to_reti			;Timer/Counter 3 Compare Match B
	jmp	to_reti			;Timer/Counter 3 Compare Match C
	jmp	to_reti			;Timer/Counter 3 Overflow

	jmp	usart1_rx			;USART1, RX Complete
	jmp	to_reti			;USART1, Data Register Empty
	jmp	to_reti			;USART1, RX Complete

	jmp	to_reti			;Two-wire Serial Interface

	jmp	to_reti			;Store Program Memory Ready

to_reti:	reti



;*****************************************************************************************
;Internal EEPROM access routines

.include	"eeprom.asm"

;*****************************************************************************************
;Floating Point Math
;
;IEEE double precision routines and stack engine
;add, sub, mul, div, sqrt, sin, cos, atan
;some 3x3 matrix and vector functions

.include	"float.asm"


;*****************************************************************************************
;Alignment and Coordinate Transformation
;
;first implementation without precession and mount fabrication errors

.include	"alignment.asm"				;two star alignment
.include	"coordinates.asm"				;conversion mount > sky coordinates
.include	"tracking_goto.asm"			;target tracking and goto
.include	"mount_coords.asm"			;transformations in mount coordinates


;*****************************************************************************************
;High Level Motor Control
;
;Speed ramping

	.EQU	MOT_INIT_POS_A	= 0		;position
	.EQU	MOT_INIT_POS_B	= 0
	.EQU	MOT_INIT_INVERT_A	= 1		;inverse direction
	.EQU	MOT_INIT_INVERT_B	= 0
	.EQU	MOT_INIT_SP_A	= 0x000000		;speed
	.EQU	MOT_INIT_SP_B	= 0x000000
	.EQU	MOT_INIT_GSP_A	= 0x000000		;goto/tracking speed
	.EQU	MOT_INIT_GSP_B	= 0x000000
	.EQU	MOT_INIT_SSP1_A	= 0x000800		;speed step 1
	.EQU	MOT_INIT_SSP1_B	= 0x000800
	.EQU	MOT_INIT_SSP2_A	= 0x004000		;speed step 2
	.EQU	MOT_INIT_SSP2_B	= 0x004000
	.EQU	MOT_INIT_SSP3_A	= 0x010000		;speed step 3
	.EQU	MOT_INIT_SSP3_B	= 0x010000
	.EQU	MOT_INIT_SSP4_A	= 0x030000		;speed step 4
	.EQU	MOT_INIT_SSP4_B	= 0x030000
	.EQU	MOT_INIT_SSP5_A	= 0x080000		;speed step 5
	.EQU	MOT_INIT_SSP5_B	= 0x080000
	.EQU	MOT_INIT_ACC_A	= 0x00c000		;acceleration
	.EQU	MOT_INIT_ACC_B	= 0x00c000

.include	"motor_ramp.asm"

;*****************************************************************************************
;Low Level Motor Control
;
;Microstepping via PWM
;Halfstepping

	.EQU	MOT_PORT_INH_A	= PORTB		;port register for inhibit signals
	.EQU	MOT_PORT_INH_B	= PORTB
	.EQU	MOT_DDR_INH_A	= DDRB		;data direction register
	.EQU	MOT_DDR_INH_B	= DDRB
	.EQU	MOT_BIT_INH_A	= 3		;port bit for inhibit of motor A
	.EQU	MOT_BIT_INH_B	= 2		;port bit for inhibit of motor B

	.EQU	MOT_PORT_A	= PORTB
	.EQU	MOT_DDR_A	= DDRB
	.EQU	MOT_BIT_PH_A1	= 4
	.EQU	MOT_BIT_PH_A2	= 5

	.EQU	MOT_PORT_B	= PORTE
	.EQU	MOT_DDR_B	= DDRE
	.EQU	MOT_BIT_PH_B1	= 2
	.EQU	MOT_BIT_PH_B2	= 3

	.EQU	MOT_INIT_MSMAX_A	= 0x018000		;microstep speed limit
	.EQU	MOT_INIT_MSMAX_B	= 0x018000
	.EQU	MOT_INIT_PER_A	= 1800		;PWM timer period (8192 per sec at 14.7456 MHz)
	.EQU	MOT_INIT_PER_B	= 1800
	.EQU	MOT_INIT_FAC_A	= 0x78		;factor for PWM on time
	.EQU	MOT_INIT_FAC_B	= 0x78
	.EQU	MOT_INIT_MIN_A	= 0x80		;minimum for PWM on time
	.EQU	MOT_INIT_MIN_B	= 0x80

.include	"motor_pwm.asm"


;*****************************************************************************************
;Serial Communication

.EQU	USART_BAUD_RATE	= 25			;38400 at 16 MHz
;.EQU	USART_BAUD_RATE	= 12			;38400 at  8 MHz

.include	"serial_com.asm"


;*****************************************************************************************
;Monitor program for debugging

monitor_msg_start:
	.db	13,13
	.db	"*****************************************",13
	.db	"* Microstepping Telescope Controller    *",13
	.db	"* Martin Cibulski     (c)    2004-02-09 *",13
	.db	"*****************************************",13
	.db	"* Monitor Commands:                     *",13
	.db	"* db <adr|lbl>            display byte  *",13
	.db	"* dw <adr|lbl>            display word  *",13
	.db	"* dl <adr|lbl>            display long  *",13
	.db	"* mb <adr|lbl> <value>    modify  byte  *",13
	.db	"* mw <adr|lbl> <value>    modify  word  *",13
	.db	"* ml <adr|lbl> <value>    modify  long  *",13
	.db	"* ports                   display ports *",13
	.db	"*****************************************",13
	.db	"* Telescope Command(s):                 *",13
	.db	"* xy <keymask> <az_speed> <alt_speed>   *",13
	.db	"*****************************************",13
	.db	"* CPU:          Atmel AVR mega128       *",13
	.db	"*               64 kWords  Pgm Flash    *",13
	.db	"*               4 kByte   data RAM      *",13
	.db	"* Microstepper: 2 x Infineon TCA 3727   *",13
	.db	"*               Dual Fullbridge with    *",13
	.db	"*               current limiting        *",13
	.db	"*****************************************",13
	.db	0,0

monitor_test_command:
	.db	"ml motapos 1234 "
	.db	0,0

.include	"Monitor.asm"


;*****************************************************************************************
;Interrupt Helper Routines
;
;was planned as a simple multitasking kernel
;but fortunately no multitasking is needed for this application :)

.include	"Multitask.asm"


;*****************************************************************************************
;Handbox Key and Display
;
;The HD66780 compatible LCD is controlled in 8 bit mode directly by the CPU.
;8 bit mode = 8 data lines + control lines ReadWrite, RegisterSelect, Enable
;
;The 8 bit data port is also used for key input (LCD's and CPU's data port switched as input).
;Therefor a fourth control output is switched to zero.
;This output is connected via 8 diodes and switches to the 8 data lines.

.equ	LCD_DATAPORT	= porta
.equ	LCD_DATAPINS	= pina
.equ	LCD_DATADDR	= ddra

.equ	LCD_CTRLPORT	= portc
.equ	LCD_CTRLPINS	= pinc
.equ	LCD_CTRLDDR	= ddrc

.equ	LCD_PIN_RW	= 2
.equ	LCD_PIN_RS	= 3
.equ	LCD_PIN_E	= 1
.equ	LCD_PIN_KEYS	= 0

.equ	LCD_LENGTH	= 8

.include	"handbox.asm"

;*****************************************************************************************
;Menue Tree
;
;The menue is defined as a tree structure with display texts and subroutine adresses.
;It is navigated in the Timer0 interrupt routine.

.include	"Menue.asm"

;*****************************************************************************************
;MAIN
;
;The program starts from here.

.CSEG

main:
	ldi	xh	,high(RAMEND)		;set stackpointer to end of RAM
	ldi	xl	,low(RAMEND)
	out	SPH	,xh
	out	SPL	,xl

	ldi	r16	,0xFF
	
main_01:
	st	X	,r16
	sbiw	xh:xl	,1
	cpi	xh	,0x01
	brcc	main_01

	clr	r16			;real time clock
	sts	clock	,r16		;128 interrupts per second
	sts	clock+1	,r16		;32 bit counter
	sts	clock+2	,r16		;> 388 days maximum 
	sts	clock+3	,r16
	sts	tgo_counter	,r16		;calculation loop counter
	sts	tgo_counter+1	,r16
	sts	tgo_counter+2	,r16
	sts	tgo_counter+3	,r16

	ldi	r16	,1		;start real time clock
	sts	clock_step	,r16

	;call	obj_calc_dir_pointers
	call	obj_test

	call	alg_init			;two star alignment function
	call	cor_init			;coordinate display
	call	tgo_init			;tracking and goto
	call	alg_calc			;alignment routine
	call	cor_calc			;current sky position
	call	tgo_calc

	call	uart_init			;serial communication line
	call	monitor_init			;debug monitor
	call	stepper_init			;stepper ramp functions
.ifdef SIMULATE
	call	monitor_test
	call	flt_test
	call	alg_test			;debug math routines
	call	cor_test
	call	tgo_calc
.endif
	clr	r0
	sts	mco_z1z2_active	,r0
	call	alg_init			;two star alignment function
	call	cor_init			;coordinate display
	call	tgo_init			;tracking and goto
	call	lcd_init			;handbox LCD
	call	mnu_init			;handbox menu
	call	stepper_init_pwm			;stepper low level and interrupts

	call	monitor_init			;debug monitor
	puts_rom	monitor_msg_start			;start message
	sei				;enable interrupts

;*****************************************************************************************
;Calculation loop
;
;Here the current sky position is calculated for displaying
;If necessary a new alignment is performed.
;Tracking will be handled here too.
;
;One loop will take about 0.1 seconds with all functions.

main_loop:	
	call	monitor			;look for command from serial line

	call	alg_calc			;alignment routine
	call	cor_calc			;current sky position
	call	tgo_calc

	rjmp	main_loop


;*****************************************************************************************
;Timer 0 Interrupt Routine
;
;Increment real time clock
;Ramp speed of Motor A
;Ramp speed of Motor B

.DSEG

clock:	.byte	4
clock_step:	.byte	1
main_loop_count:	.byte	4

.CSEG

irq_timer_0:
	push	r16
	in	r16	,SREG
	push	r16
	push	r17

	lds	r17	,clock_step
	lds	r16	,clock+3
	add	r16	,r17
	sts	clock+3	,r16
	clr	r17
	lds	r16	,clock+2
	adc	r16	,r17
	sts	clock+2	,r16
	lds	r16	,clock+1
	adc	r16	,r17
	sts	clock+1	,r16
	lds	r16	,clock
	adc	r16	,r17
	sts	clock	,r16

	call	mt_enter_interrupt		;enable nested interrupts

	ldi	yh	,high(mot_a)		;calculations for motor A
	ldi	yl	,low(mot_a)
	call	mot_ramp

	ldi	yh	,high(mot_b)		;calculations for motor A
	ldi	yl	,low(mot_b)
	call	mot_ramp

	call	lcd_irq

	jmp	mt_leave_interrupt


;*****************************************************************************************
;Timer 1 Interrupt Routine (Motor A)
;
;Set phase for this (just beginning) micro/halfstep
;calculate and set PWM values for next micro/halfstep

irq_timer_1:
	push	r16
	in	r16	,SREG
	push	r16
	push	r17

	lds	r16	,mot_a_ph_1
	lds	r17	,mot_a_ph_2
	sbrs	r16	,0
	cbi	MOT_PORT_A	,MOT_BIT_PH_A1
	sbrc	r16	,0
	sbi	MOT_PORT_A	,MOT_BIT_PH_A1
	sbrs	r17	,0
	cbi	MOT_PORT_A	,MOT_BIT_PH_A2
	sbrc	r17	,0
	sbi	MOT_PORT_A	,MOT_BIT_PH_A2

	call	mt_enter_interrupt		;enable nested interrupts

	ldi	yh	,high(mot_a)		;calculations for motor A
	ldi	yl	,low(mot_a)
	ldi	r21	,OCR1AL+0x20
	ldi	r22	,OCR1BL+0x20
	ldi	r23	,OCR1CL
	call	mot_calc

	jmp	mt_leave_interrupt


;************************************************************
;Timer 3 Interrupt Routine (Motor B)
;
;Set phase for this (just beginning) micro/halfstep
;calculate and set PWM values for next micro/halfstep

irq_timer_3:
	push	r16
	in	r16	,SREG
	push	r16
	push	r17

	lds	r16	,mot_b_ph_1
	lds	r17	,mot_b_ph_2
	sbrs	r16	,0
	cbi	MOT_PORT_B	,MOT_BIT_PH_B1
	sbrc	r16	,0
	sbi	MOT_PORT_B	,MOT_BIT_PH_B1
	sbrs	r17	,0
	cbi	MOT_PORT_B	,MOT_BIT_PH_B2
	sbrc	r17	,0
	sbi	MOT_PORT_B	,MOT_BIT_PH_B2

	call	mt_enter_interrupt		;enable nested interrupts

	ldi	yh	,high(mot_b)		;calculations for motor B
	ldi	yl	,low(mot_b)
	ldi	r21	,OCR3AL
	ldi	r22	,OCR3BL
	ldi	r23	,OCR3CL
	call	mot_calc

	jmp	mt_leave_interrupt


;*****************************************************************************************
;Object database
.include	"object_db.asm"
.include	"ngc_ic_12.txt"

;*****************************************************************************************
