

;************************************************************
;	Mega 128 Scope Controller Program
;	(c) Martin Cibulski
;************************************************************


;************************************************************
;	Stepper Motor Calculations
;	called from the timer 0 interrupt routine
;	128 times per second
;
;Parameters	yh:yl	adress of motor data block
;	r21	RAM adress of OCR1AL/OCR3AL
;	r22	RAM adress of OCR1BL/OCR3BL
;	r23	RAM adress of OCR1CL/OCR3CL
.CSEG

mot_calc:
	cli
	ldd	r2	,Y+MOT_POS		;motor position
	ldd	r3	,Y+MOT_POS+1
	ldd	r4	,Y+MOT_POS+2
	ldd	r5	,Y+MOT_POS+3
	ldd	r6	,Y+MOT_POS+4
	ldd	r7	,Y+MOT_POS+5
	sei

	cli
	ldd	r8	,Y+MOT_SP		;motor speed
	ldd	r9	,Y+MOT_SP+1
	ldd	r10	,Y+MOT_SP+2
	ldd	r11	,Y+MOT_SP+3
	sei

	ldd	r16	,Y+MOT_HSMODE		;halfstep mode set ?
	tst	r16
	brne	mot_calc_halfstep_mode

	add	r7	,r11		;microstep mode, add speed to position
	adc	r6	,r10
	adc	r5	,r9
	adc	r4	,r8		;always 0x00 or 0xFF
	adc	r3	,r8
	adc	r2	,r8
	rjmp	mot_calc_store_position

mot_calc_halfstep_mode:
	ldi	r16	,0x20		;one halfstep = 32 microsteps
	ldi	r17	,0

	tst	r8			;speed positive ?
	brmi	mot_calc_hs_negative		;no

	add	r5	,r16		;add one halfstep
	adc	r4	,r17
	adc	r3	,r17
	adc	r2	,r17
	rjmp	mot_calc_store_position

mot_calc_hs_negative:
	sub	r5	,r16		;substract one halfstep
	sbc	r4	,r17
	sbc	r3	,r17
	sbc	r2	,r17

mot_calc_store_position:	
	cli
	std	Y+MOT_POS	,r2		;new position
	std	Y+MOT_POS+1	,r3
	std	Y+MOT_POS+2	,r4
	std	Y+MOT_POS+3	,r5
	std	Y+MOT_POS+4	,r6
	std	Y+MOT_POS+5	,r7
	sei

	mov	r20	,r8		;save sign of speed
	tst	r8			;positive speed ?
	brpl	mot_calc_speed_positive		;yes

	com	r11			;make speed positive
	com	r10
	com	r9
	com	r8
	ldi	r16	,0xFF
	sub	r11	,r16
	sbc	r10	,r16
	sbc	r9	,r16
	sbc	r8	,r16

mot_calc_speed_positive:
	ldd	r16	,Y+MOT_MSMAX+3		;speed > max microstep speed ?
	cp	r11	,r16
	ldd	r16	,Y+MOT_MSMAX+2
	cpc	r10	,r16
	ldd	r16	,Y+MOT_MSMAX+1
	cpc	r9	,r16
	ldd	r16	,Y+MOT_MSMAX
	cpc	r8	,r16
	brcs	mot_move_microstep		;no, normal PWM microstepping

	mov	r16	,r5		;switching to halfstep mode allowed ?
	andi	r16	,0x1F
	cpi	r16	,0x10
	brne	mot_move_microstep		;no, continue microstepping

	rjmp	mot_move_halfstep			;yes, set halfstep timer period

;************************************************************
;	PWM Timing for next microstep
mot_move_microstep:
	mov	r16	,r5		;microstep position (0-255)
	;lsr	r16			;convert to (0-127)
	ldi	r18	,4		;size of table entry
	mul	r16	,r18
	ldi	zl	,low(mot_table<<1)	;table base adress
	ldi	zh	,high(mot_table<<1)
	add	zl	,r0
	adc	zh	,r1

	lpm	r12	,Z+		;PWM on time 1
	lpm	r13	,Z+		;PWM on time 2
	lpm	r14	,Z+		;Phase 1
	ldd	r15	,Y+MOT_INVERTED
	eor	r14	,r15
	lpm	r15	,Z+		;Phase 2

	std	Y+MOT_PH1	,r14		;store phase information
	std	Y+MOT_PH2	,r15		;will be set at next IRQ

	ldd	r15	,Y+MOT_PWMMIN
	ldd	r14	,Y+MOT_PWMMIN+1

	ldd	r17	,Y+MOT_PWMPER		;set PWM timer period
	ldd	r16	,y+MOT_PWMPER+1
	mov	xl	,r21		;adress of output compare register A
	clr	xh
	adiw	xh:xl	,2		;adress after OC register B
	st	-X	,r17		;OCRxA high byte first
	st	-X	,r16		;OCRxA low byte

	ldd	r18	,Y+MOT_PWMFAC		;PULSE = MOT_PWMMIN + MOT_PWMFAC * TABLE_VALUE / 64
	mul	r18	,r12		;r12 = TABLE_VALUE (coil A)
	asr	r1			; / 64
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	add	r0	,r14		; + MOT_PWMMIN
	adc	r1	,r15
	ldd	r17	,Y+MOT_PWMPER		;Timer period
	ldd	r16	,y+MOT_PWMPER+1
	sub	r16	,r0		;timer compare value for switching on
	sbc	r17	,r1		;= period - PULSE
	brcc	mot_move_microstep_01		;not negative ?

	clr	r16
	clr	r17

mot_move_microstep_01:
	mov	xl	,r22		;adress of output compare register B
	clr	xh
	adiw	xh:xl	,2		;adress after OC register B
	st	-X	,r17		;OCRxB high byte first
	st	-X	,r16		;OCRxB low byte

	ldd	r18	,Y+MOT_PWMFAC		;PULSE = MOT_PWMMIN + MOT_PWMFAC * TABLE_VALUE / 64
	mul	r18	,r13		;r13 = TABLE_VALUE (coil B)
	asr	r1			; / 64
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	asr	r1
	ror	r0
	add	r0	,r14		; + MOT_PWMMIN
	adc	r1	,r15
	ldd	r17	,Y+MOT_PWMPER		;Timer period
	ldd	r16	,Y+MOT_PWMPER+1
	sub	r16	,r0		;timer compare value for switching on
	sbc	r17	,r1		;= period - PULSE
	brcc	mot_move_microstep_02		;not negative ?

	clr	r16
	clr	r17

mot_move_microstep_02:
	mov	xl	,r23		;adress of output compare register C
	clr	xh
	adiw	xh:xl	,2		;adress after OC register C
	st	-X	,r17		;OCRxC high byte first
	st	-X	,r16		;OCRxC low byte

	clr	r16			;set microstep mode for next interrupt
	std	Y+MOT_HSMODE	,r16
	ret

;************************************************************
;	Halfstepping
;	Set IRQ timer period for one halfstep to go
mot_move_halfstep:
	mov	r16	,r5		;microstep position (0-255)
	lsr	r16			;convert to (0-127)
	lsr	r16			;convert to (0- 63)
	lsr	r16			;convert to (0- 31)
	lsr	r16			;convert to (0- 15)
	lsr	r16			;convert to (0-  7)
	ldi	r18	,6		;size of table entry
	mul	r16	,r18
	ldi	zl	,low(mot_table_halfstep<<1)	;table base adress
	ldi	zh	,high(mot_table_halfstep<<1)
	add	zl	,r0
	adc	zh	,r1

	lpm	r12	,Z+		;PWM on time 1
	lpm	r13	,Z+		;PWM on time 2
	lpm	r14	,Z+		;Phase 1
	lpm	r15	,Z+		;Phase 2

	sbrc	r20	,7		;negative speed ?
	lpm	r14	,Z+		;use backward phases
	sbrc	r20	,7
	lpm	r15	,Z+

	ldd	r17	,Y+MOT_INVERTED
	eor	r14	,r17

	std	Y+MOT_PH1	,r14		;store phase information
	std	Y+MOT_PH2	,r15		;will be set at next IRQ

	ldd	r17	,Y+MOT_PWMPER		;set PWM timer period
	ldd	r16	,y+MOT_PWMPER+1
mot_move_hs_shift:
	add	r16	,r16		;timer period *= 2
	adc	r17	,r17
	add	r10	,r10		;speed *= 2
	adc	r9	,r9
	bst	r9	,5		;speed >= 0x200000 (1 halfstep per IRQ)
	brtc	mot_move_hs_shift

	add	r10	,r10		;speed *= 2 (0x40 <= speed <= 0x7F)
	adc	r9	,r9
	add	r10	,r10		;speed *= 2 (0x80 <= speed <= 0xFF)
	adc	r9	,r9
	ldi	r18	,128
	sub	r9	,r18		;speed -= 128 (0 <= speed <= 0x7F)
	ldi	zl	,low(invert_table<<1)	;table base adress
	ldi	zh	,high(invert_table<<1)
	clr	r18
	add	zl	,r9		;add index
	adc	zh	,r18
	lpm	r18	,Z		;factor for timer period

	mul	r18	,r16		;multiply PWM period low byte
	mov	r2	,r0
	mov	r3	,r1
	clr	r4
	mul	r18	,r17		;multiply PWM period high byte
	add	r3	,r0
	adc	r4	,r1

	add	r3	,r3
	adc	r4	,r4

	mov	xl	,r21		;adress of output compare register A
	clr	xh
	adiw	xh:xl	,2		;adress after OC register B
	st	-X	,r4		;OCRxA high byte first
	st	-X	,r3		;OCRxA low byte

	ldi	r18	,0xFF
	eor	r12	,r18
	eor	r13	,r18

	mov	xl	,r22		;adress of output compare register B
	clr	xh
	adiw	xh:xl	,2		;adress after OC register B
	st	-X	,r12		;OCRxB high byte first
	st	-X	,r12		;OCRxB low byte

	mov	xl	,r23		;adress of output compare register C
	clr	xh
	adiw	xh:xl	,2		;adress after OC register C
	st	-X	,r13		;OCRxC high byte first
	st	-X	,r13		;OCRxC low byte

	ldi	r16	,0xFF		;set halfstep mode for next interrupt
	std	Y+MOT_HSMODE	,r16
	ret

;************************************************************
;	Stepper Motor Initialization
;	Low Level
;************************************************************

.CSEG

stepper_init_pwm:

	ldi	r16	,0xFF		;disable PWM (0xFFFF into output compare registers B,C)
	out	OCR1BH	,r16
	out	OCR1BL	,r16
	sts	OCR1CH	,r16
	sts	OCR1CL	,r16
	sts	OCR3BH	,r16
	sts	OCR3BL	,r16
	sts	OCR3CH	,r16
	sts	OCR3CL	,r16

;	Motor A

	ldi	r16	,0b11110000		;OC1C, OC1B must be output pins
	out	MOT_DDR_A	,r16

	ldi	yl	,low(mot_a)
	ldi	yh	,high(mot_a)

	ldi	r16	,byte4(MOT_INIT_MSMAX_A)	;max speed for microstepping of motor A
	ldi	r17	,byte3(MOT_INIT_MSMAX_A)
	ldi	r18	,high (MOT_INIT_MSMAX_A)
	ldi	r19	,low  (MOT_INIT_MSMAX_A)
	std	Y+MOT_MSMAX	,r16
	std	Y+MOT_MSMAX+1	,r17
	std	Y+MOT_MSMAX+2	,r18
	std	Y+MOT_MSMAX+3	,r19

	ldi	r16	,high(MOT_INIT_PER_A-1)	;PWM timer period
	ldi	r17	,low(MOT_INIT_PER_A-1)
	std	Y+MOT_PWMPER	,r16
	std	Y+MOT_PWMPER+1	,r17

	ldi	r16	,MOT_INIT_FAC_A		;factor for PWM on time
	std	Y+MOT_PWMFAC	,r16

	ldi	r16	,high(MOT_INIT_MIN_A)	;minimum PWM on time
	ldi	r17	,low(MOT_INIT_MIN_A)
	std	Y+MOT_PWMMIN	,r16
	std	Y+MOT_PWMMIN+1	,r17

	ldi	r16	,0		;Timer start value
	out	TCNT1H	,r16
	out	TCNT1L	,r16

	ldd	r17	,Y+MOT_PWMPER
	ldd	r16	,Y+MOT_PWMPER+1
	out	OCR1AH	,r17
	out	OCR1AL	,r16

	ldi	r16	,0b00101011		;Clear OCnB/C on Compare Match, fast PWM ??? 
	out	TCCR1A	,r16
	ldi	r16	,0b00011001		;Fast PWM, no Prescaling
	out	TCCR1B	,r16

;	Motor B

	ldi	r16	,0b00111100		;OC1C, OC1B must be output pins
	out	MOT_DDR_B	,r16

	ldi	r16	,0b11111111		;Motor control port ???
	sts	MOT_DDR_B	,r16
	sts	MOT_PORT_B	,r16

	ldi	yl	,low(mot_b)
	ldi	yh	,high(mot_b)

	ldi	r16	,byte4(MOT_INIT_MSMAX_B)	;max speed for microstepping of motor B
	ldi	r17	,byte3(MOT_INIT_MSMAX_B)
	ldi	r18	,high (MOT_INIT_MSMAX_B)
	ldi	r19	,low  (MOT_INIT_MSMAX_B)
	std	Y+MOT_MSMAX	,r16
	std	Y+MOT_MSMAX+1	,r17
	std	Y+MOT_MSMAX+2	,r18
	std	Y+MOT_MSMAX+3	,r19

	ldi	r16	,high(MOT_INIT_PER_B-1)	;PWM timer period
	ldi	r17	,low(MOT_INIT_PER_B-1)
	std	Y+MOT_PWMPER	,r16
	std	Y+MOT_PWMPER+1	,r17

	ldi	r16	,MOT_INIT_FAC_B		;factor for PWM on time
	std	Y+MOT_PWMFAC	,r16

	ldi	r16	,high(MOT_INIT_MIN_B)	;minimum PWM on time
	ldi	r17	,low(MOT_INIT_MIN_B)
	std	Y+MOT_PWMMIN	,r16
	std	Y+MOT_PWMMIN+1	,r17

	ldi	r16	,0		;Timer start value
	sts	TCNT3H	,r16
	sts	TCNT3L	,r16

	ldd	r17	,Y+MOT_PWMPER
	ldd	r16	,y+MOT_PWMPER+1
	sts	OCR3AH	,r17
	sts	OCR3AL	,r16

	ldi	r16	,0b00101011		;Clear OCnB/C on Compare Match, fast PWM ??? 
	sts	TCCR3A	,r16
	ldi	r16	,0b00011001		;Fast PWM, no Prescaling
	sts	TCCR3B	,r16

;	Inhibit
	sbi	MOT_DDR_INH_A	,MOT_BIT_INH_A
	sbi	MOT_DDR_INH_B	,MOT_BIT_INH_B
	sbi	MOT_PORT_INH_A	,MOT_BIT_INH_A
	sbi	MOT_PORT_INH_B	,MOT_BIT_INH_B

;	Speed Ramping Interrupt
.ifdef SIMULATE
	ldi	r16	,255
	out	OCR0	,r16
	;ldi	r16	,0b01001110		;run to 0xFF, no OC pin, prescaling 256
	ldi	r16	,(1<<WGM01)|(1<<CS02)|(1<<CS01)	;run to OCR0, no OC pin, prescaling 256
	out	TCCR0	,r16
	ldi	r16	,0b00000000		;synchronous operation from CPU clock
	out	ASSR	,r16
	rjmp	stepper_init_pwm_enable

.else
	ldi	r16	,1<<AS0		;asynchronous operation from 32768 Hz
	out	ASSR	,r16

	ldi	r16	,(1<<WGM01)|(1<<CS00)	;run to OCR0, no OC pin, no prescaling
	out	TCCR0	,r16

	ldi	r16	,(1<<OCF0)
	out	TIFR	,r16

	ldi	r16	,255
	out	OCR0	,r16

stepper_init_pwm_wait:
	in	r16	,ASSR
	sbrc	r16	,OCR0UB
	rjmp	stepper_init_pwm_wait

.endif

stepper_init_pwm_enable:
;	Enable Interrupts
	;ldi	r16	,(1<<OCIE0)
	ldi	r16	,(1<<OCIE1A) | (1<<OCIE0)
	out	TIMSK	,r16

	ldi	r16	,1<<OCIE3A
	sts	ETIMSK	,r16

	ret

      


;************************************************************
;
.include	"invert_table.txt"
.include	"mot_table.txt"

mot_table_halfstep:
.db	  0,255,1,1,0,1
.db	255,255,1,1,1,1
.db	255,  0,1,0,1,1
.db	255,255,1,0,1,0
.db	  0,255,0,0,1,0
.db	255,255,0,0,0,0
.db	255,  0,0,1,0,0
.db	255,255,0,1,0,1

mot_table_halfstep_16:
.db	  0,255,0,1,0,1
.db	255,255,1,1,1,1
.db	255,255,1,1,1,1
.db	255,  0,1,1,1,1
.db	255,  0,1,1,1,1
.db	255,255,1,0,1,0
.db	255,255,1,0,1,0
.db	  0,255,1,0,1,0
.db	  0,255,1,0,1,0
.db	255,255,0,0,0,0
.db	255,255,0,0,0,0
.db	255,  0,0,0,0,0
.db	255,  0,0,0,0,0
.db	255,255,0,1,0,1
.db	255,255,0,1,0,1
.db	  0,255,0,1,0,1

;************************************************************
